'use client';

import { useState, useEffect, useCallback } from 'react';
import axios from 'axios';
import './Home.css';
import ImageCarousel from '../components/ImageCarousel';

// ———————————————————————————————————————————————
// 🔸 Product Skeleton Loader
// ———————————————————————————————————————————————
const ProductSkeleton = () => (
  <div className="product-skeleton" role="status" aria-label="Loading product">
    <div className="skeleton-line title" />
    <div className="skeleton-line short" />
    <div className="skeleton-line medium" />
    <div className="skeleton-line long" />
  </div>
);

// ———————————————————————————————————————————————
// 🔸 Product Card — FIXED IMAGE FALLBACK TO ONLINE URL
// ———————————————————————————————————————————————
const ProductCard = ({ product }) => {
  const price = parseFloat(product?.price);
  const formattedPrice = isNaN(price) ? 'N/A' : price.toFixed(2);

  // Use online fallback image if no valid image
  const imageUrl =
    product.image && product.image.startsWith('http')
      ? product.image
      : 'https://images.unsplash.com/photo-1607082349566-187342175e2f?auto=format&fit=crop&w=600&h=600&q=80';

  return (
    <article
      className="product-card"
      itemScope
      itemType="http://schema.org/Product"
      aria-labelledby={`product-name-${product._id || product.name}`}
    >
      <div className="product-image">
        <img
          src={imageUrl}
          alt={product.name || 'Product image'}
          loading="lazy"
          onError={(e) => {
            e.target.src = 'https://images.unsplash.com/photo-1523275335684-37898b6baf30?auto=format&fit=crop&w=600&h=600&q=80';
          }}
        />
      </div>
      <div className="product-info">
        <h2
          id={`product-name-${product._id || product.name}`}
          className="product-name"
          itemProp="name"
        >
          {product.name ?? 'Untitled Product'}
        </h2>
        <p className="product-desc" itemProp="description">
          {product.description && product.description.length > 100
            ? `${product.description.substring(0, 100)}...`
            : product.description ?? 'No description available.'}
        </p>
        <p className="product-price" itemProp="offers" itemScope itemType="http://schema.org/Offer">
          <span itemProp="priceCurrency" content="PKR">Rs.</span>
          <span itemProp="price">{formattedPrice}</span>
        </p>
      </div>
    </article>
  );
};

// ———————————————————————————————————————————————
// 🔸 Main Component — Home (Vulvera – With Social Links)
// ———————————————————————————————————————————————
export default function Home() {
  const [products, setProducts] = useState([]);
  const [status, setStatus] = useState({ loading: true, error: null });

  const fetchProducts = useCallback(async () => {
    setStatus({ loading: true, error: null });
    try {
      const { data } = await axios.get('/api/products', { timeout: 8000 });
      setProducts(Array.isArray(data) ? data : []);
      setStatus({ loading: false, error: null });
    } catch (err) {
      console.error('Product fetch failed:', err);
      setStatus({
        loading: false,
        error:
          err.code === 'ECONNABORTED'
            ? 'Request timed out. Please try again.'
            : err.message === 'Network Error'
            ? 'Unable to connect to the server. Please check your connection.'
            : 'Failed to load products. Please try again later.',
      });
    }
  }, []);

  useEffect(() => {
    fetchProducts();
  }, [fetchProducts]);

  return (
    <main className="home-page" aria-label="Vulvera Home Page">
      {/* Hero Carousel Section */}
      <section className="hero-section" aria-label="Featured Collection Carousel">
        <ImageCarousel />
      </section>

      {/* Header Text */}
      <header className="home-header" aria-labelledby="collection-heading">
        <h1 id="collection-heading">Discover Our Collection</h1>
        <p>
          Every item is thoughtfully designed, ethically sourced, and crafted to inspire.
          Welcome to quality that lasts.
        </p>
      </header>

      {/* Product Grid / States */}
      {status.loading && (
        <section className="product-grid" aria-busy="true">
          {Array.from({ length: 8 }).map((_, i) => (
            <ProductSkeleton key={i} />
          ))}
        </section>
      )}

      {status.error && (
        <section className="error-section" aria-live="polite">
          <div className="error-box">
            <h2>We ran into a problem</h2>
            <p>{status.error}</p>
            <button onClick={fetchProducts} aria-label="Retry loading products">
              Try Again
            </button>
          </div>
        </section>
      )}

      {!status.loading && !status.error && products.length > 0 && (
        <section className="product-grid" aria-label="Product catalog">
          {products.map((p) => (
            <ProductCard key={p._id || p.name} product={p} />
          ))}
        </section>
      )}

      {!status.loading && !status.error && products.length === 0 && (
        <section className="empty-state" aria-label="No products available">
          <div className="empty-box">
            <h2>No Products Found</h2>
            <p>We’re working hard to bring you amazing items soon.</p>
          </div>
        </section>
      )}

      {/* Floating Social Buttons */}
      <div className="social-float-container">
        {/* WhatsApp */}
        <a
          href="https://wa.me/923395936453"
          className="social-float-btn whatsapp"
          target="_blank"
          rel="noopener noreferrer"
          aria-label="Chat with us on WhatsApp"
        >
          <img
            src="https://upload.wikimedia.org/wikipedia/commons/6/6b/WhatsApp.svg"
            alt=""
            className="social-icon"
            aria-hidden="true"
          />
        </a>

        {/* Instagram — UPDATE HANDLE IF NEEDED */}
        <a
          href="https://www.instagram.com/velveraluxe/"
          className="social-float-btn instagram"
          target="_blank"
          rel="noopener noreferrer"
          aria-label="Follow us on Instagram"
        >
          <svg className="social-icon" viewBox="0 0 24 24" aria-hidden="true">
            <path
              fill="currentColor"
              d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.28-.073 1.688-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.28.058 1.689.072 4.949.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.949 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.28-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"
            />
          </svg>
        </a>

        {/* TikTok — UPDATE HANDLE IF NEEDED */}
        <a
          href="https://www.tiktok.com/@velveraluxe"
          className="social-float-btn tiktok"
          target="_blank"
          rel="noopener noreferrer"
          aria-label="Follow us on TikTok"
        >
          <svg className="social-icon" viewBox="0 0 24 24" aria-hidden="true">
            <path
              fill="currentColor"
              d="M12.53.02C13.84 0 15.14.01 16.44 0c.08 1.53.63 3.09 1.75 4.17 1.12 1.11 2.7 1.62 4.24 1.79v4.03c-1.44-.05-2.89-.35-4.2-.97-.57-.26-1.1-.59-1.62-.93-.01 2.92.01 5.84-.02 8.75-.08 1.4-.54 2.79-1.35 3.94-1.31 1.92-3.58 3.17-5.91 3.21-1.43.08-2.86-.31-4.08-1.03-2.02-1.19-3.44-3.37-3.65-5.71-.02-.5-.03-1-.01-1.49.18-1.9 1.12-3.72 2.58-4.96 1.66-1.44 3.98-2.13 6.15-1.72.02 1.48-.04 2.96-.04 4.44-.99-.32-2.15-.23-3.02.37-.63.41-1.11 1.04-1.36 1.75-.21.51-.15 1.07-.14 1.61.24 1.64 1.82 3.02 3.5 2.87 1.12-.01 2.19-.66 2.77-1.61.19-.33.4-.67.41-1.06.1-1.79.06-3.57.07-5.36.01-4.03-.01-8.05.02-12.07z"
            />
          </svg>
        </a>
      </div>
    </main>
  );
}