import { useState, useEffect } from 'react';
import axios from 'axios';

export default function Admin() {
  const [categories, setCategories] = useState([]);
  const [products, setProducts] = useState([]);
  const [orders, setOrders] = useState([]);

  const [newCategoryName, setNewCategoryName] = useState('');
  const [newProduct, setNewProduct] = useState({
    name: '',
    description: '',
    price: '',
    category: '',
    imageUrl: '', // Optional: external URL
  });
  const [imageFile, setImageFile] = useState(null); // New: for file upload
  const [imagePreview, setImagePreview] = useState(null);

  // Fetch all data on load
  useEffect(() => {
    const fetchData = async () => {
      try {
        const [catRes, prodRes, orderRes] = await Promise.all([
          axios.get('/api/categories'),
          axios.get('/api/products'),
          axios.get('/api/orders'),
        ]);
        setCategories(catRes.data);
        setProducts(prodRes.data);
        setOrders(orderRes.data);
      } catch (err) {
        console.error('Failed to load admin ', err);
        alert('Failed to load dashboard data. Check console.');
      }
    };
    fetchData();
  }, []);

  // Handle image URL change
  const handleImageUrlChange = (e) => {
    const url = e.target.value;
    setNewProduct({ ...newProduct, imageUrl: url });
    setImagePreview(url);
    setImageFile(null); // Clear file if using URL
  };

  // Handle file selection
  const handleImageFileChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      setImageFile(file);
      setNewProduct({ ...newProduct, imageUrl: '' }); // Clear URL if using file
      const reader = new FileReader();
      reader.onload = () => {
        setImagePreview(reader.result);
      };
      reader.readAsDataURL(file);
    }
  };

  // Add Category
  const handleAddCategory = async (e) => {
    e.preventDefault();
    if (!newCategoryName.trim()) return;
    try {
      const res = await axios.post('/api/categories', { name: newCategoryName });
      setCategories([...categories, res.data]);
      setNewCategoryName('');
    } catch (err) {
      alert('Failed to add category');
    }
  };

  // Add Product
  const handleAddProduct = async (e) => {
    e.preventDefault();
    const { name, description, price, category } = newProduct;
    if (!name || !description || !price || !category) {
      alert('All fields except image are required');
      return;
    }

    try {
      let imageUrl = newProduct.imageUrl;

      // If a file is selected, upload it first (multipart)
      if (imageFile) {
        const formData = new FormData();
        formData.append('image', imageFile);

        // Upload image to your image upload endpoint (you must implement this!)
        const uploadRes = await axios.post('/api/upload', formData, {
          headers: { 'Content-Type': 'multipart/form-data' },
        });
        imageUrl = uploadRes.data.url; // Assumes your API returns { url: "..." }
      }

      // Now create product with final imageUrl (could be empty string if none provided)
      const productData = {
        name,
        description,
        price: parseFloat(price),
        category,
        imageUrl: imageUrl || '', // explicitly allow empty
      };

      const res = await axios.post('/api/products', productData);
      setProducts([...products, res.data]);
      // Reset form
      setNewProduct({ name: '', description: '', price: '', category: '', imageUrl: '' });
      setImageFile(null);
      setImagePreview(null);
    } catch (err) {
      console.error('Add product error:', err);
      alert('Failed to add product. Make sure /api/upload is implemented.');
    }
  };

  // Delete Product
  const handleDeleteProduct = async (id) => {
    if (!window.confirm('Are you sure you want to delete this product?')) return;
    try {
      await axios.delete(`/api/products/${id}`);
      setProducts(products.filter((p) => p._id !== id));
    } catch (err) {
      alert('Failed to delete product');
    }
  };

  // Delete Category
  const handleDeleteCategory = async (id) => {
    if (!window.confirm('Are you sure? Products will remain but lose category.')) return;
    try {
      await axios.delete(`/api/categories/${id}`);
      setCategories(categories.filter((c) => c._id !== id));
    } catch (err) {
      alert('Failed to delete category');
    }
  };

  // Update Order Status
  const handleUpdateOrderStatus = async (orderId, status) => {
    try {
      await axios.patch(`/api/orders/${orderId}/status`, { status });
      setOrders(orders.map((order) => (order._id === orderId ? { ...order, status } : order)));
    } catch (err) {
      alert('Failed to update order');
    }
  };

  // Logout handler
  const handleLogout = () => {
    localStorage.removeItem('isAdminAuthenticated');
    window.location.href = '/';
  };

  // Inline Styles (unchanged)
  const containerStyle = {
    maxWidth: '1000px',
    margin: '0 auto',
    padding: '20px',
    fontFamily: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
  };

  const headerStyle = {
    textAlign: 'center',
    fontSize: '2.2rem',
    color: '#ef4444',
    margin: '20px 0',
  };

  const logoutButtonStyle = {
    backgroundColor: '#ef4444',
    color: 'white',
    border: 'none',
    padding: '8px 16px',
    borderRadius: '6px',
    cursor: 'pointer',
    fontWeight: '600',
    marginBottom: '20px',
    marginLeft: 'auto',
    display: 'block',
  };

  const cardStyle = {
    backgroundColor: '#fff',
    borderRadius: '10px',
    boxShadow: '0 4px 12px rgba(0,0,0,0.08)',
    padding: '20px',
    marginBottom: '24px',
  };

  const formGroupStyle = {
    marginBottom: '16px',
  };

  const inputStyle = {
    width: '100%',
    padding: '10px',
    borderRadius: '6px',
    border: '1px solid #ddd',
    fontSize: '1rem',
    boxSizing: 'border-box',
  };

  const textareaStyle = {
    ...inputStyle,
    minHeight: '80px',
    resize: 'vertical',
  };

  const selectStyle = {
    ...inputStyle,
    height: '42px',
  };

  const buttonPrimaryStyle = {
    backgroundColor: '#4f46e5',
    color: 'white',
    border: 'none',
    padding: '10px 20px',
    borderRadius: '6px',
    cursor: 'pointer',
    fontSize: '1rem',
    fontWeight: '600',
  };

  const buttonDangerStyle = {
    ...buttonPrimaryStyle,
    backgroundColor: '#ef4444',
  };

  const buttonSuccessStyle = {
    ...buttonPrimaryStyle,
    backgroundColor: '#10b981',
  };

  const productListStyle = {
    display: 'grid',
    gridTemplateColumns: 'repeat(auto-fill, minmax(240px, 1fr))',
    gap: '20px',
    marginTop: '16px',
  };

  const productCardStyle = {
    border: '1px solid #eee',
    borderRadius: '8px',
    overflow: 'hidden',
    boxShadow: '0 2px 6px rgba(0,0,0,0.05)',
  };

  const productImageStyle = {
    width: '100%',
    height: '160px',
    objectFit: 'cover',
    backgroundColor: '#f9fafb',
  };

  const productInfoStyle = {
    padding: '12px',
  };

  return (
    <div style={containerStyle}>
      {/* Logout Button */}
      <button onClick={handleLogout} style={logoutButtonStyle}>
        🔐 Logout
      </button>

      <h1 style={headerStyle}>👨‍💼 Velvera Admin Dashboard</h1>

      {/* Add Category */}
      <div style={cardStyle}>
        <h2 style={{ margin: '0 0 16px' }}>➕ Add New Category</h2>
        <form onSubmit={handleAddCategory}>
          <div style={formGroupStyle}>
            <input
              type="text"
              placeholder="Category name"
              value={newCategoryName}
              onChange={(e) => setNewCategoryName(e.target.value)}
              style={inputStyle}
            />
          </div>
          <button type="submit" style={buttonPrimaryStyle}>
            Add Category
          </button>
        </form>
      </div>

      {/* Categories List */}
      <div style={cardStyle}>
        <h2 style={{ margin: '0 0 16px' }}>📁 Categories ({categories.length})</h2>
        {categories.length === 0 ? (
          <p>No categories yet</p>
        ) : (
          <ul style={{ listStyle: 'none', padding: 0 }}>
            {categories.map((cat) => (
              <li
                key={cat._id}
                style={{
                  display: 'flex',
                  justifyContent: 'space-between',
                  padding: '10px 0',
                  borderBottom: '1px solid #f1f1f1',
                }}
              >
                <span>{cat.name}</span>
                <button
                  onClick={() => handleDeleteCategory(cat._id)}
                  style={{ ...buttonDangerStyle, padding: '4px 10px', fontSize: '0.85rem' }}
                >
                  Delete
                </button>
              </li>
            ))}
          </ul>
        )}
      </div>

      {/* Add Product */}
      <div style={cardStyle}>
        <h2 style={{ margin: '0 0 16px' }}>📦 Add New Product</h2>
        <form onSubmit={handleAddProduct}>
          <div style={formGroupStyle}>
            <input
              type="text"
              placeholder="Product name"
              value={newProduct.name}
              onChange={(e) => setNewProduct({ ...newProduct, name: e.target.value })}
              style={inputStyle}
              required
            />
          </div>
          <div style={formGroupStyle}>
            <textarea
              placeholder="Description"
              value={newProduct.description}
              onChange={(e) => setNewProduct({ ...newProduct, description: e.target.value })}
              style={textareaStyle}
              required
            />
          </div>
          <div style={formGroupStyle}>
            <input
              type="number"
              placeholder="Price (e.g., 29.99)"
              value={newProduct.price}
              onChange={(e) => setNewProduct({ ...newProduct, price: e.target.value })}
              min="0"
              step="0.01"
              style={inputStyle}
              required
            />
          </div>
          <div style={formGroupStyle}>
            <select
              value={newProduct.category}
              onChange={(e) => setNewProduct({ ...newProduct, category: e.target.value })}
              style={selectStyle}
              required
            >
              <option value="">Select Category</option>
              {categories.map((cat) => (
                <option key={cat._id} value={cat._id}>
                  {cat.name}
                </option>
              ))}
            </select>
          </div>

          {/* Image URL Option */}
          {/* Image Options */}
<div style={formGroupStyle}>
  <h3 style={{ fontSize: '1.1rem', margin: '0 0 12px', color: '#4b5563' }}>
    🖼️ Product Image (Optional)
  </h3>

  {/* URL Input */}
  <div style={{ marginBottom: '16px' }}>
    <label style={{ display: 'block', marginBottom: '6px', fontSize: '0.95rem' }}>
      Image URL (optional):
    </label>
    <input
      type="url"
      placeholder="e.g., https://example.com/product.jpg"
      value={newProduct.imageUrl}
      onChange={handleImageUrlChange}
      style={inputStyle}
      disabled={imageFile !== null} // Disable if file is selected
    />
  </div>

  {/* File Upload */}
  <div>
    <label style={{ display: 'block', marginBottom: '6px', fontSize: '0.95rem' }}>
      Upload from device:
    </label>
    <input
      type="file"
      accept="image/*"
      onChange={handleImageFileChange}
      style={{
        width: '100%',
        padding: '8px',
        border: '2px dashed #d1d5db',
        borderRadius: '6px',
        backgroundColor: '#f9fafb',
        cursor: 'pointer',
      }}
    />
    <p style={{ fontSize: '0.85rem', color: '#6b7280', marginTop: '6px' }}>
      Supports JPG, PNG, GIF (Max 5MB)
    </p>
  </div>
</div>

          <button type="submit" style={buttonPrimaryStyle}>
            ➕ Add Product
          </button>
        </form>
      </div>

      {/* Products List */}
      <div style={cardStyle}>
        <h2 style={{ margin: '0 0 16px' }}>🛍️ All Products ({products.length})</h2>
        {products.length === 0 ? (
          <p>No products yet</p>
        ) : (
          <div style={productListStyle}>
            {products.map((product) => (
              <div key={product._id} style={productCardStyle}>
                <img
                  src={product.imageUrl || 'https://via.placeholder.com/240x160?text=No+Image'}
                  alt={product.name}
                  style={productImageStyle}
                />
                <div style={productInfoStyle}>
                  <h3 style={{ fontSize: '1.1rem', margin: '0 0 6px' }}>{product.name}</h3>
                  <p style={{ fontSize: '0.9rem', color: '#555', margin: '0 0 8px' }}>
                    ${product.price.toFixed(2)}
                  </p>
                  <button
                    onClick={() => handleDeleteProduct(product._id)}
                    style={{ ...buttonDangerStyle, padding: '6px 12px', fontSize: '0.85rem' }}
                  >
                    Delete
                  </button>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {/* Orders Section */}
      <div style={cardStyle}>
        <h2 style={{ margin: '0 0 16px' }}>💳 Payment Verification ({orders.length} orders)</h2>
        {orders.length === 0 ? (
          <p>No orders yet</p>
        ) : (
          orders.map((order) => (
            <div
              key={order._id}
              style={{
                border: '1px solid #e2e8f0',
                borderRadius: '8px',
                padding: '16px',
                marginBottom: '16px',
                backgroundColor: '#fafafa',
              }}
            >
              <p>
                <strong>User ID:</strong> {order.userId?.substring(0, 8) || 'N/A'}
              </p>
              <p>
                <strong>Status:</strong>
                <span
                  style={{
                    color:
                      order.status === 'Confirmed'
                        ? '#10b981'
                        : order.status === 'Rejected'
                        ? '#ef4444'
                        : '#f59e0b',
                    marginLeft: '8px',
                    fontWeight: '500',
                  }}
                >
                  {order.status}
                </span>
              </p>
              <p>
                <strong>Products:</strong>
              </p>
              <ul style={{ paddingLeft: '20px', marginBottom: '12px' }}>
                {order.products?.map((item, i) => (
                  <li key={i}>
                    {item.productId?.name || 'Unknown Product'} × {item.quantity}
                  </li>
                ))}
              </ul>
              {order.paymentScreenshot && (
                <>
                  <p>
                    <strong>Payment Proof:</strong>
                  </p>
                  <img
                    src={order.paymentScreenshot}
                    alt="Payment proof"
                    style={{
                      width: '150px',
                      height: '150px',
                      objectFit: 'cover',
                      borderRadius: '6px',
                      border: '1px solid #ddd',
                      marginTop: '8px',
                    }}
                    onError={(e) => {
                      e.target.style.display = 'none';
                    }}
                  />
                </>
              )}
              <div style={{ marginTop: '12px' }}>
                <button
                  onClick={() => handleUpdateOrderStatus(order._id, 'Confirmed')}
                  style={{
                    ...buttonSuccessStyle,
                    marginRight: '10px',
                    opacity: order.status === 'Confirmed' ? 0.6 : 1,
                  }}
                  disabled={order.status === 'Confirmed'}
                >
                  ✅ Confirm
                </button>
                <button
                  onClick={() => handleUpdateOrderStatus(order._id, 'Rejected')}
                  style={{
                    ...buttonDangerStyle,
                    opacity: order.status === 'Rejected' ? 0.6 : 1,
                  }}
                  disabled={order.status === 'Rejected'}
                >
                  ❌ Reject
                </button>
              </div>
            </div>
          ))
        )}
      </div>
    </div>
  );
}
