// Navbar.jsx — updated
import { useState, useEffect } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import axios from 'axios';
import './Navbar.css';

export default function Navbar() {
  const [categories, setCategories] = useState([]);
  const [search, setSearch] = useState('');
  const [menuOpen, setMenuOpen] = useState(false);
  const navigate = useNavigate();

  // Auth states
  const isAdmin = localStorage.getItem('isAdminAuthenticated') === 'true';
  const isClient = localStorage.getItem('isClientAuthenticated') === 'true';
  const clientEmail = localStorage.getItem('clientEmail');

  // Fetch categories
  useEffect(() => {
    const fetchCategories = async () => {
      try {
        const res = await axios.get('/api/categories');
        setCategories(res.data);
      } catch (err) {
        console.error('Error loading categories:', err);
      }
    };
    fetchCategories();
  }, []);

  const handleSearch = (e) => {
    e.preventDefault();
    if (search.trim()) {
      navigate(`/search?q=${encodeURIComponent(search.trim())}`);
      setSearch('');
    }
  };

  const handleLogout = () => {
    localStorage.removeItem('isAdminAuthenticated');
    localStorage.removeItem('isClientAuthenticated');
    localStorage.removeItem('clientEmail');
    navigate('/');
    window.location.reload();
  };

  return (
    <header className="navbar">
      <div className="navbar-container">
        {/* Logo */}
        <div className="navbar-logo">
          <Link to="/">Velvera</Link>
        </div>

        {/* Search Bar */}
        <form onSubmit={handleSearch} className="navbar-search">
          <input
            type="text"
            placeholder="Search supplements, vitamins..."
            value={search}
            onChange={(e) => setSearch(e.target.value)}
          />
          <button type="submit">🔍</button>
        </form>

        {/* Right Side Icons */}
        <div className="navbar-icons">
          <Link to="/contact" className="contact-btn">Contact Us</Link> {/* 👈 Added */}
          <Link to="/checkout" className="icon">🛒</Link>

          {!isAdmin && !isClient && (
            <Link to="/login" className="icon">👤 Login</Link>
          )}

          {isClient && (
            <div className="client-section">
              <span className="client-email">Hi, {clientEmail?.split('@')[0]}</span>
              <button className="logout-btn" onClick={handleLogout}>Logout</button>
            </div>
          )}

          {isAdmin && (
            <div className="admin-section">
              <Link to="/admin" className="admin-link">👨‍💼 Admin</Link>
              <button className="logout-btn" onClick={handleLogout}>Logout</button>
            </div>
          )}

          <button
            className="menu-toggle"
            onClick={() => setMenuOpen(!menuOpen)}
          >
            ☰
          </button>
        </div>
      </div>

      {/* Category Bar */}
      <nav className="navbar-categories">
        {categories.length > 0 ? (
          categories.map((cat) => (
            <Link key={cat._id} to={`/category/${cat.slug || cat._id}`}>
              {cat.name}
            </Link>
          ))
        ) : (
          <div className="hero-loading-bar">
            <span className="loading-text">Loading categories...</span>
          </div>
        )}
      </nav>

      {/* Mobile Menu */}
      {menuOpen && (
        <div className="navbar-mobile-menu">
          <Link to="/" onClick={() => setMenuOpen(false)}>Home</Link>
          <Link to="/contact" onClick={() => setMenuOpen(false)}>Contact</Link>
          <Link to="/checkout" onClick={() => setMenuOpen(false)}>Checkout</Link>
          {isAdmin && (
            <Link to="/admin" onClick={() => setMenuOpen(false)}>Admin</Link>
          )}
          {(isClient || isAdmin) && (
            <button className="logout-btn mobile" onClick={handleLogout}>
              Logout
            </button>
          )}
        </div>
      )}
    </header>
  );
}