import * as React$1 from 'react';

interface ApiKey {
    created_at: string;
    id: string;
    name: string;
}

interface DomainApiOptions {
    name: string;
    region?: string;
    custom_return_path?: string;
}

declare const RESEND_ERROR_CODES_BY_KEY: {
    readonly missing_required_field: 422;
    readonly invalid_idempotency_key: 400;
    readonly invalid_idempotent_request: 409;
    readonly concurrent_idempotent_requests: 409;
    readonly invalid_access: 422;
    readonly invalid_parameter: 422;
    readonly invalid_region: 422;
    readonly rate_limit_exceeded: 429;
    readonly missing_api_key: 401;
    readonly invalid_api_key: 403;
    readonly suspended_api_key: 403;
    readonly invalid_from_address: 403;
    readonly validation_error: 403;
    readonly not_found: 404;
    readonly method_not_allowed: 405;
    readonly application_error: 500;
    readonly internal_server_error: 500;
};
type RESEND_ERROR_CODE_KEY = keyof typeof RESEND_ERROR_CODES_BY_KEY;
type Response<T> = ({
    data: T;
    error: null;
} | {
    error: ErrorResponse;
    data: null;
}) & {
    headers: Record<string, string> | null;
};
type ErrorResponse = {
    message: string;
    statusCode: number | null;
    name: RESEND_ERROR_CODE_KEY;
};
type Tag$1 = {
    name: string;
    value: string;
};

interface EmailApiAttachment {
    content?: string | Buffer;
    filename?: string | false | undefined;
    path?: string;
    content_type?: string;
    content_id?: string;
}
interface EmailApiOptions {
    from?: string;
    to: string | string[];
    subject?: string;
    region?: string;
    headers?: Record<string, string>;
    html?: string;
    text?: string;
    bcc?: string | string[];
    cc?: string | string[];
    reply_to?: string | string[];
    topic_id?: string | null;
    scheduled_at?: string;
    tags?: Tag$1[];
    attachments?: EmailApiAttachment[];
    template?: {
        id: string;
        variables?: Record<string, string | number>;
    };
}

interface GetOptions {
    query?: Record<string, unknown>;
    headers?: HeadersInit;
}

interface IdempotentRequest {
    /**
     * Unique key that ensures the same operation is not processed multiple times.
     * Allows for safe retries without duplicating operations.
     * If provided, will be sent as the `Idempotency-Key` header.
     */
    idempotencyKey?: string;
}

type PaginationOptions = {
    /**
     * Maximum number of items to return (1-100, default: 20)
     */
    limit?: number;
} & ({
    /**
     * Get items after this cursor (cannot be used with 'before')
     */
    after?: string;
    before?: never;
} | {
    /**
     * Get items before this cursor (cannot be used with 'after')
     */
    before?: string;
    after?: never;
});
type PaginatedData<Data> = {
    object: 'list';
    data: Data;
    has_more: boolean;
};

interface PatchOptions {
    query?: {
        [key: string]: unknown;
    };
    headers?: HeadersInit;
}

interface PostOptions {
    query?: {
        [key: string]: unknown;
    };
    headers?: HeadersInit;
}

interface PutOptions {
    query?: {
        [key: string]: unknown;
    };
    headers?: HeadersInit;
}

type RequireAtLeastOne<T> = {
    [K in keyof T]-?: Required<Pick<T, K>> & Partial<Pick<T, Exclude<keyof T, K>>>;
}[keyof T];

interface CreateApiKeyOptions {
    name: string;
    permission?: 'full_access' | 'sending_access';
    domain_id?: string;
}
interface CreateApiKeyRequestOptions extends PostOptions {
}
interface CreateApiKeyResponseSuccess {
    token: string;
    id: string;
}
type CreateApiKeyResponse = Response<CreateApiKeyResponseSuccess>;

type ListApiKeysOptions = PaginationOptions;
type ListApiKeysResponseSuccess = {
    object: 'list';
    has_more: boolean;
    data: Pick<ApiKey, 'name' | 'id' | 'created_at'>[];
};
type ListApiKeysResponse = Response<ListApiKeysResponseSuccess>;

type RemoveApiKeyResponseSuccess = {};
type RemoveApiKeyResponse = Response<RemoveApiKeyResponseSuccess>;

interface EmailRenderOptions$1 {
    /**
     * The React component used to write the message.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    react: React$1.ReactNode;
    /**
     * The HTML version of the message.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    html: string;
    /**
     * The plain text version of the message.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    text: string;
}
interface EmailTemplateOptions {
    template: {
        id: string;
        variables?: Record<string, string | number>;
    };
}
interface CreateEmailBaseOptionsWithTemplate extends Omit<CreateEmailBaseOptions, 'from' | 'subject'> {
    from?: string;
    subject?: string;
}
interface CreateEmailBaseOptions {
    /**
     * Filename and content of attachments (max 40mb per email)
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    attachments?: Attachment$1[];
    /**
     * Blind carbon copy recipient email address. For multiple addresses, send as an array of strings.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    bcc?: string | string[];
    /**
     * Carbon copy recipient email address. For multiple addresses, send as an array of strings.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    cc?: string | string[];
    /**
     * Sender email address. To include a friendly name, use the format `"Your Name <sender@domain.com>"`
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    from: string;
    /**
     * Custom headers to add to the email.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    headers?: Record<string, string>;
    /**
     * Reply-to email address. For multiple addresses, send as an array of strings.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    replyTo?: string | string[];
    /**
     * Email subject.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    subject: string;
    /**
     * Email tags
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    tags?: Tag[];
    /**
     * Recipient email address. For multiple addresses, send as an array of strings. Max 50.
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    to: string | string[];
    /**
     * The id of the topic you want to send to
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    topicId?: string | null;
    /**
     * Schedule email to be sent later.
     * The date should be in ISO 8601 format (e.g: 2024-08-05T11:52:01.858Z).
     *
     * @link https://resend.com/docs/api-reference/emails/send-email#body-parameters
     */
    scheduledAt?: string;
}
type CreateEmailOptions = ((RequireAtLeastOne<EmailRenderOptions$1> & CreateEmailBaseOptions) & {
    template?: never;
}) | ((EmailTemplateOptions & CreateEmailBaseOptionsWithTemplate) & {
    react?: never;
    html?: never;
    text?: never;
});
interface CreateEmailRequestOptions extends PostOptions, IdempotentRequest {
}
interface CreateEmailResponseSuccess {
    /** The ID of the newly created email. */
    id: string;
}
type CreateEmailResponse = Response<CreateEmailResponseSuccess>;
interface Attachment$1 {
    /** Content of an attached file. */
    content?: string | Buffer;
    /** Name of attached file. */
    filename?: string | false | undefined;
    /** Path where the attachment file is hosted */
    path?: string;
    /** Optional content type for the attachment, if not set will be derived from the filename property */
    contentType?: string;
    /**
     * Optional content ID for the attachment, to be used as a reference in the HTML content.
     * If set, this attachment will be sent as an inline attachment and you can reference it in the HTML content using the `cid:` prefix.
     */
    contentId?: string;
}
type Tag = {
    /**
     * The name of the email tag. It can only contain ASCII letters (a–z, A–Z), numbers (0–9), underscores (_), or dashes (-). It can contain no more than 256 characters.
     */
    name: string;
    /**
     * The value of the email tag. It can only contain ASCII letters (a–z, A–Z), numbers (0–9), underscores (_), or dashes (-). It can contain no more than 256 characters.
     */
    value: string;
};

type CreateBatchOptions = CreateEmailOptions[];
interface CreateBatchRequestOptions extends PostOptions, IdempotentRequest {
    /**
     * @default 'strict'
     */
    batchValidation?: 'strict' | 'permissive';
}
type CreateBatchSuccessResponse<Options extends CreateBatchRequestOptions = CreateBatchRequestOptions> = {
    data: {
        /** The ID of the newly created email. */
        id: string;
    }[];
} & (Options['batchValidation'] extends 'permissive' ? {
    /**
     * Only present when header "x-batch-validation" is set to 'permissive'.
     */
    errors: {
        /**
         * The index of the failed email in the batch
         */
        index: number;
        /**
         * The error message for the failed email
         */
        message: string;
    }[];
} : Record<string, never>);
type CreateBatchResponse<Options extends CreateBatchRequestOptions> = Response<CreateBatchSuccessResponse<Options>>;

interface Broadcast {
    id: string;
    name: string;
    segment_id: string | null;
    audience_id: string | null;
    from: string | null;
    subject: string | null;
    reply_to: string[] | null;
    preview_text: string | null;
    status: 'draft' | 'sent' | 'queued';
    created_at: string;
    scheduled_at: string | null;
    sent_at: string | null;
    topic_id?: string | null;
    html: string | null;
    text: string | null;
}

interface EmailRenderOptions {
    /**
     * The React component used to write the message.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    react: React$1.ReactNode;
    /**
     * The HTML version of the message.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    html: string;
    /**
     * The plain text version of the message.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    text: string;
}
interface SegmentOptions {
    /**
     * The id of the segment you want to send to
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    segmentId: string;
    /**
     * @deprecated Use segmentId instead
     */
    audienceId: string;
}
interface CreateBroadcastBaseOptions {
    /**
     * The name of the broadcast
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    name?: string;
    /**
     * A short snippet of text displayed as a preview in recipients' inboxes, often shown below or beside the subject line.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    previewText?: string;
    /**
     * Sender email address. To include a friendly name, use the format `"Your Name <sender@domain.com>"`
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    from: string;
    /**
     * Reply-to email address. For multiple addresses, send as an array of strings.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    replyTo?: string | string[];
    /**
     * Email subject.
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    subject: string;
    /**
     * The id of the topic you want to send to
     *
     * @link https://resend.com/docs/api-reference/broadcasts/create#body-parameters
     */
    topicId?: string | null;
}
type CreateBroadcastOptions = RequireAtLeastOne<EmailRenderOptions> & RequireAtLeastOne<SegmentOptions> & CreateBroadcastBaseOptions;
interface CreateBroadcastRequestOptions extends PostOptions {
}
interface CreateBroadcastResponseSuccess {
    /** The ID of the newly sent broadcasts. */
    id: string;
}
type CreateBroadcastResponse = Response<CreateBroadcastResponseSuccess>;

interface GetBroadcastResponseSuccess extends Broadcast {
    object: 'broadcast';
}
type GetBroadcastResponse = Response<GetBroadcastResponseSuccess>;

type ListBroadcastsOptions = PaginationOptions;
type ListBroadcastsResponseSuccess = {
    object: 'list';
    has_more: boolean;
    data: Pick<Broadcast, 'id' | 'name' | 'audience_id' | 'segment_id' | 'status' | 'created_at' | 'scheduled_at' | 'sent_at'>[];
};
type ListBroadcastsResponse = Response<ListBroadcastsResponseSuccess>;

interface RemoveBroadcastResponseSuccess extends Pick<Broadcast, 'id'> {
    object: 'broadcast';
    deleted: boolean;
}
type RemoveBroadcastResponse = Response<RemoveBroadcastResponseSuccess>;

interface SendBroadcastBaseOptions {
    /**
     * Schedule email to be sent later.
     * The date should be in ISO 8601 format (e.g: 2024-08-05T11:52:01.858Z)
     * or relative time (eg: in 2 days).
     *
     * @link https://resend.com/docs/api-reference/broadcasts/send#body-parameters
     */
    scheduledAt?: string;
}
type SendBroadcastOptions = SendBroadcastBaseOptions;
interface SendBroadcastRequestOptions extends PostOptions {
}
interface SendBroadcastResponseSuccess {
    /** The ID of the sent broadcast. */
    id: string;
}
type SendBroadcastResponse = Response<SendBroadcastResponseSuccess>;

interface UpdateBroadcastResponseSuccess {
    id: string;
}
type UpdateBroadcastOptions = {
    name?: string;
    segmentId?: string;
    /**
     * @deprecated Use segmentId instead
     */
    audienceId?: string;
    from?: string;
    html?: string;
    react?: React.ReactNode;
    text?: string;
    subject?: string;
    replyTo?: string[];
    previewText?: string;
    topicId?: string | null;
};
type UpdateBroadcastResponse = Response<UpdateBroadcastResponseSuccess>;

interface Contact {
    created_at: string;
    id: string;
    email: string;
    first_name?: string;
    last_name?: string;
    unsubscribed: boolean;
}
type SelectingField = {
    /**
     * The contact id.
     *
     * @link https://resend.com/docs/api-reference/contacts/delete-contact#body-parameters
     */
    id: string;
    /**
     * The contact email.
     *
     * @link https://resend.com/docs/api-reference/contacts/delete-contact#body-parameters
     */
    email?: undefined | null;
} | {
    /**
     * The contact id.
     *
     * @link https://resend.com/docs/api-reference/contacts/delete-contact#body-parameters
     */
    id?: undefined | null;
    /**
     * The contact email.
     *
     * @link https://resend.com/docs/api-reference/contacts/delete-contact#body-parameters
     */
    email: string;
};

interface CreateContactPropertiesOptions {
    [key: string]: string | number | null;
}
interface CreateContactOptions {
    audienceId?: string;
    email: string;
    unsubscribed?: boolean;
    firstName?: string;
    lastName?: string;
    properties?: CreateContactPropertiesOptions;
}
interface CreateContactRequestOptions extends PostOptions {
}
interface CreateContactResponseSuccess extends Pick<Contact, 'id'> {
    object: 'contact';
}
type CreateContactResponse = Response<CreateContactResponseSuccess>;

type GetContactOptions = string | ({
    audienceId?: string;
} & SelectingField);
type ContactPropertyValue = {
    type: 'string';
    value: string;
} | {
    type: 'number';
    value: number;
};
interface ContactProperties$1 {
    [key: string]: ContactPropertyValue;
}
interface GetContactResponseSuccess extends Pick<Contact, 'id' | 'email' | 'created_at' | 'first_name' | 'last_name' | 'unsubscribed'> {
    object: 'contact';
    properties: ContactProperties$1;
}
type GetContactResponse = Response<GetContactResponseSuccess>;

type ListAudienceContactsOptions = {
    audienceId: string;
} & PaginationOptions;
type ListContactsOptions = PaginationOptions & {
    audienceId?: string;
};
interface ListContactsResponseSuccess {
    object: 'list';
    data: Contact[];
    has_more: boolean;
}
type ListContactsResponse = Response<ListContactsResponseSuccess>;

type RemoveContactsResponseSuccess = {
    object: 'contact';
    deleted: boolean;
    contact: string;
};
type RemoveContactOptions = string | (SelectingField & {
    audienceId?: string;
});
type RemoveContactsResponse = Response<RemoveContactsResponseSuccess>;

interface UpdateContactPropertiesOptions {
    [key: string]: string | number | null;
}
type UpdateContactOptions = {
    audienceId?: string;
    unsubscribed?: boolean;
    firstName?: string;
    lastName?: string;
    properties?: UpdateContactPropertiesOptions;
} & SelectingField;
type UpdateContactResponseSuccess = Pick<Contact, 'id'> & {
    object: 'contact';
};
type UpdateContactResponse = Response<UpdateContactResponseSuccess>;

type DomainRegion = 'us-east-1' | 'eu-west-1' | 'sa-east-1' | 'ap-northeast-1';
type DomainNameservers = 'Amazon Route 53' | 'Cloudflare' | 'Digital Ocean' | 'GoDaddy' | 'Google Domains' | 'Namecheap' | 'Unidentified' | 'Vercel';
type DomainStatus = 'pending' | 'verified' | 'failed' | 'temporary_failure' | 'not_started';
type DomainRecords = DomainSpfRecord | DomainDkimRecord | ReceivingRecord;
interface DomainSpfRecord {
    record: 'SPF';
    name: string;
    value: string;
    type: 'MX' | 'TXT';
    ttl: string;
    status: DomainStatus;
    routing_policy?: string;
    priority?: number;
    proxy_status?: 'enable' | 'disable';
}
interface DomainDkimRecord {
    record: 'DKIM';
    name: string;
    value: string;
    type: 'CNAME' | 'TXT';
    ttl: string;
    status: DomainStatus;
    routing_policy?: string;
    priority?: number;
    proxy_status?: 'enable' | 'disable';
}
interface ReceivingRecord {
    record: 'Receiving';
    name: string;
    value: string;
    type: 'MX';
    ttl: string;
    status: DomainStatus;
    priority: number;
}
interface Domain {
    id: string;
    name: string;
    status: DomainStatus;
    created_at: string;
    region: DomainRegion;
}

interface CreateDomainOptions {
    name: string;
    region?: DomainRegion;
    customReturnPath?: string;
}
interface CreateDomainRequestOptions extends PostOptions {
}
interface CreateDomainResponseSuccess extends Pick<Domain, 'name' | 'id' | 'status' | 'created_at' | 'region'> {
    records: DomainRecords[];
}
type CreateDomainResponse = Response<CreateDomainResponseSuccess>;

interface GetDomainResponseSuccess extends Pick<Domain, 'id' | 'name' | 'created_at' | 'region' | 'status'> {
    object: 'domain';
    records: DomainRecords[];
}
type GetDomainResponse = Response<GetDomainResponseSuccess>;

type ListDomainsOptions = PaginationOptions;
type ListDomainsResponseSuccess = {
    data: Domain[];
    object: 'list';
    has_more: boolean;
};
type ListDomainsResponse = Response<ListDomainsResponseSuccess>;

type RemoveDomainsResponseSuccess = Pick<Domain, 'id'> & {
    object: 'domain';
    deleted: boolean;
};
type RemoveDomainsResponse = Response<RemoveDomainsResponseSuccess>;

interface UpdateDomainsOptions {
    id: string;
    clickTracking?: boolean;
    openTracking?: boolean;
    tls?: 'enforced' | 'opportunistic';
}
type UpdateDomainsResponseSuccess = Pick<Domain, 'id'> & {
    object: 'domain';
};
type UpdateDomainsResponse = Response<UpdateDomainsResponseSuccess>;

type VerifyDomainsResponseSuccess = Pick<Domain, 'id'> & {
    object: 'domain';
};
type VerifyDomainsResponse = Response<VerifyDomainsResponseSuccess>;

interface Attachment {
    id: string;
    filename?: string;
    size: number;
    content_type: string;
    content_disposition: 'inline' | 'attachment';
    content_id?: string;
    download_url: string;
    expires_at: string;
}

interface GetAttachmentOptions {
    emailId: string;
    id: string;
}
type GetAttachmentResponseSuccess = {
    object: 'attachment';
} & Attachment;
type GetAttachmentResponse = Response<GetAttachmentResponseSuccess>;

type ListAttachmentsOptions = PaginationOptions & {
    emailId: string;
};
interface ListAttachmentsApiResponse {
    object: 'list';
    has_more: boolean;
    data: Attachment[];
}
interface ListAttachmentsResponseSuccess {
    object: 'list';
    has_more: boolean;
    data: Attachment[];
}
type ListAttachmentsResponse = Response<ListAttachmentsResponseSuccess>;

interface CancelEmailResponseSuccess {
    object: 'email';
    id: string;
}
type CancelEmailResponse = Response<CancelEmailResponseSuccess>;

interface GetEmailResponseSuccess {
    bcc: string[] | null;
    cc: string[] | null;
    created_at: string;
    from: string;
    html: string | null;
    id: string;
    last_event: 'bounced' | 'canceled' | 'clicked' | 'complained' | 'delivered' | 'delivery_delayed' | 'failed' | 'opened' | 'queued' | 'scheduled' | 'sent';
    reply_to: string[] | null;
    subject: string;
    text: string | null;
    tags?: {
        name: string;
        value: string;
    }[];
    to: string[];
    topic_id?: string | null;
    scheduled_at: string | null;
    object: 'email';
}
type GetEmailResponse = Response<GetEmailResponseSuccess>;

interface UpdateEmailOptions {
    id: string;
    scheduledAt: string;
}
interface UpdateEmailResponseSuccess {
    id: string;
    object: 'email';
}
type UpdateEmailResponse = Response<UpdateEmailResponseSuccess>;

interface GetReceivingEmailResponseSuccess {
    object: 'email';
    id: string;
    to: string[];
    from: string;
    created_at: string;
    subject: string;
    bcc: string[] | null;
    cc: string[] | null;
    reply_to: string[] | null;
    html: string | null;
    text: string | null;
    headers: Record<string, string>;
    message_id: string;
    attachments: Array<{
        id: string;
        filename: string;
        size: number;
        content_type: string;
        content_id: string;
        content_disposition: string;
    }>;
}
type GetReceivingEmailResponse = Response<GetReceivingEmailResponseSuccess>;

type ListReceivingEmailsOptions = PaginationOptions;
type ListReceivingEmail = Omit<GetReceivingEmailResponseSuccess, 'html' | 'text' | 'headers' | 'object'>;
interface ListReceivingEmailsResponseSuccess {
    object: 'list';
    has_more: boolean;
    data: ListReceivingEmail[];
}
type ListReceivingEmailsResponse = {
    data: ListReceivingEmailsResponseSuccess;
    error: null;
} | {
    data: null;
    error: ErrorResponse;
};

declare class ApiKeys {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateApiKeyOptions, options?: CreateApiKeyRequestOptions): Promise<CreateApiKeyResponse>;
    list(options?: ListApiKeysOptions): Promise<ListApiKeysResponse>;
    remove(id: string): Promise<RemoveApiKeyResponse>;
}

declare class Batch {
    private readonly resend;
    constructor(resend: Resend);
    send<Options extends CreateBatchRequestOptions>(payload: CreateBatchOptions, options?: Options): Promise<CreateBatchResponse<Options>>;
    create<Options extends CreateBatchRequestOptions>(payload: CreateBatchOptions, options?: Options): Promise<CreateBatchResponse<Options>>;
}

declare class Broadcasts {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateBroadcastOptions, options?: CreateBroadcastRequestOptions): Promise<SendBroadcastResponse>;
    send(id: string, payload?: SendBroadcastOptions): Promise<SendBroadcastResponse>;
    list(options?: ListBroadcastsOptions): Promise<ListBroadcastsResponse>;
    get(id: string): Promise<GetBroadcastResponse>;
    remove(id: string): Promise<RemoveBroadcastResponse>;
    update(id: string, payload: UpdateBroadcastOptions): Promise<UpdateBroadcastResponse>;
}

type StringBaseContactProperty = {
    type: 'string';
    fallbackValue: string | null;
};
type NumberBaseContactProperty = {
    type: 'number';
    fallbackValue: number | null;
};
type ContactProperty = {
    id: string;
    object: 'contact_property';
    createdAt: string;
    key: string;
} & (StringBaseContactProperty | NumberBaseContactProperty);

type StringBaseContactPropertyOptions = {
    type: 'string';
    fallbackValue?: string | null;
};
type NumberBaseContactPropertyOptions = {
    type: 'number';
    fallbackValue?: number | null;
};
type CreateContactPropertyOptions = {
    key: string;
} & (StringBaseContactPropertyOptions | NumberBaseContactPropertyOptions);
type CreateContactPropertyResponseSuccess = Pick<ContactProperty, 'id' | 'object'>;
type CreateContactPropertyResponse = Response<CreateContactPropertyResponseSuccess>;

type RemoveContactPropertyResponseSuccess = Pick<ContactProperty, 'id' | 'object'> & {
    deleted: boolean;
};
type RemoveContactPropertyResponse = Response<RemoveContactPropertyResponseSuccess>;

type GetContactPropertyResponse = Response<ContactProperty>;

type ListContactPropertiesOptions = PaginationOptions;
type ListContactPropertiesResponse = Response<{
    object: 'list';
    has_more: boolean;
    data: ContactProperty[];
}>;

type UpdateContactPropertyOptions = {
    id: string;
    fallbackValue?: string | number | null;
};
type UpdateContactPropertyResponseSuccess = Pick<ContactProperty, 'id' | 'object'>;
type UpdateContactPropertyResponse = Response<UpdateContactPropertyResponseSuccess>;

declare class ContactProperties {
    private readonly resend;
    constructor(resend: Resend);
    create(options: CreateContactPropertyOptions): Promise<CreateContactPropertyResponse>;
    list(options?: ListContactPropertiesOptions): Promise<ListContactPropertiesResponse>;
    get(id: string): Promise<GetContactPropertyResponse>;
    update(payload: UpdateContactPropertyOptions): Promise<UpdateContactPropertyResponse>;
    remove(id: string): Promise<RemoveContactPropertyResponse>;
}

type ContactSegmentsBaseOptions = {
    contactId: string;
    email?: never;
} | {
    contactId?: never;
    email: string;
};

type AddContactSegmentOptions = ContactSegmentsBaseOptions & {
    segmentId: string;
};
interface AddContactSegmentResponseSuccess {
    id: string;
}
type AddContactSegmentResponse = Response<AddContactSegmentResponseSuccess>;

interface Segment {
    created_at: string;
    id: string;
    name: string;
}

type ListContactSegmentsOptions = PaginationOptions & ContactSegmentsBaseOptions;
type ListContactSegmentsResponseSuccess = PaginatedData<Segment[]>;
type ListContactSegmentsResponse = Response<ListContactSegmentsResponseSuccess>;

type RemoveContactSegmentOptions = ContactSegmentsBaseOptions & {
    segmentId: string;
};
interface RemoveContactSegmentResponseSuccess {
    id: string;
    deleted: boolean;
}
type RemoveContactSegmentResponse = Response<RemoveContactSegmentResponseSuccess>;

declare class ContactSegments {
    private readonly resend;
    constructor(resend: Resend);
    list(options: ListContactSegmentsOptions): Promise<ListContactSegmentsResponse>;
    add(options: AddContactSegmentOptions): Promise<AddContactSegmentResponse>;
    remove(options: RemoveContactSegmentOptions): Promise<RemoveContactSegmentResponse>;
}

interface ListContactTopicsBaseOptions {
    id?: string;
    email?: string;
}
type ListContactTopicsOptions = ListContactTopicsBaseOptions & PaginationOptions;
interface ContactTopic {
    id: string;
    name: string;
    description: string | null;
    subscription: 'opt_in' | 'opt_out';
}
type ListContactTopicsResponseSuccess = PaginatedData<ContactTopic[]>;
type ListContactTopicsResponse = Response<ListContactTopicsResponseSuccess>;

interface UpdateContactTopicsBaseOptions {
    id?: string;
    email?: string;
}
interface UpdateContactTopicsOptions extends UpdateContactTopicsBaseOptions {
    topics: {
        id: string;
        subscription: 'opt_in' | 'opt_out';
    }[];
}
interface UpdateContactTopicsResponseSuccess {
    id: string;
}
type UpdateContactTopicsResponse = Response<UpdateContactTopicsResponseSuccess>;

declare class ContactTopics {
    private readonly resend;
    constructor(resend: Resend);
    update(payload: UpdateContactTopicsOptions): Promise<UpdateContactTopicsResponse>;
    list(options: ListContactTopicsOptions): Promise<ListContactTopicsResponse>;
}

declare class Contacts {
    private readonly resend;
    readonly topics: ContactTopics;
    readonly segments: ContactSegments;
    constructor(resend: Resend);
    create(payload: CreateContactOptions, options?: CreateContactRequestOptions): Promise<CreateContactResponse>;
    list(options?: ListContactsOptions | ListAudienceContactsOptions): Promise<ListContactsResponse>;
    get(options: GetContactOptions): Promise<GetContactResponse>;
    update(options: UpdateContactOptions): Promise<UpdateContactResponse>;
    remove(payload: RemoveContactOptions): Promise<RemoveContactsResponse>;
}

declare class Domains {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateDomainOptions, options?: CreateDomainRequestOptions): Promise<CreateDomainResponse>;
    list(options?: ListDomainsOptions): Promise<ListDomainsResponse>;
    get(id: string): Promise<GetDomainResponse>;
    update(payload: UpdateDomainsOptions): Promise<UpdateDomainsResponse>;
    remove(id: string): Promise<RemoveDomainsResponse>;
    verify(id: string): Promise<VerifyDomainsResponse>;
}

declare class Attachments$1 {
    private readonly resend;
    constructor(resend: Resend);
    get(options: GetAttachmentOptions): Promise<GetAttachmentResponse>;
    list(options: ListAttachmentsOptions): Promise<ListAttachmentsResponse>;
}

type ListEmailsOptions = PaginationOptions;
type ListEmail = Omit<GetEmailResponseSuccess, 'html' | 'text' | 'tags' | 'object'>;
type ListEmailsResponseSuccess = {
    object: 'list';
    has_more: boolean;
    data: ListEmail[];
};
type ListEmailsResponse = Response<ListEmailsResponseSuccess>;

declare class Attachments {
    private readonly resend;
    constructor(resend: Resend);
    get(options: GetAttachmentOptions): Promise<GetAttachmentResponse>;
    list(options: ListAttachmentsOptions): Promise<ListAttachmentsResponse>;
}

declare class Receiving {
    private readonly resend;
    readonly attachments: Attachments;
    constructor(resend: Resend);
    get(id: string): Promise<GetReceivingEmailResponse>;
    list(options?: ListReceivingEmailsOptions): Promise<ListReceivingEmailsResponse>;
}

declare class Emails {
    private readonly resend;
    readonly attachments: Attachments$1;
    readonly receiving: Receiving;
    constructor(resend: Resend);
    send(payload: CreateEmailOptions, options?: CreateEmailRequestOptions): Promise<CreateEmailResponse>;
    create(payload: CreateEmailOptions, options?: CreateEmailRequestOptions): Promise<CreateEmailResponse>;
    get(id: string): Promise<GetEmailResponse>;
    list(options?: ListEmailsOptions): Promise<ListEmailsResponse>;
    update(payload: UpdateEmailOptions): Promise<UpdateEmailResponse>;
    cancel(id: string): Promise<CancelEmailResponse>;
}

interface CreateSegmentOptions {
    name: string;
}
interface CreateSegmentRequestOptions extends PostOptions {
}
interface CreateSegmentResponseSuccess extends Pick<Segment, 'name' | 'id'> {
    object: 'segment';
}
type CreateSegmentResponse = Response<CreateSegmentResponseSuccess>;

interface GetSegmentResponseSuccess extends Pick<Segment, 'id' | 'name' | 'created_at'> {
    object: 'segment';
}
type GetSegmentResponse = Response<GetSegmentResponseSuccess>;

type ListSegmentsOptions = PaginationOptions;
type ListSegmentsResponseSuccess = {
    object: 'list';
    data: Segment[];
    has_more: boolean;
};
type ListSegmentsResponse = Response<ListSegmentsResponseSuccess>;

interface RemoveSegmentResponseSuccess extends Pick<Segment, 'id'> {
    object: 'segment';
    deleted: boolean;
}
type RemoveSegmentResponse = Response<RemoveSegmentResponseSuccess>;

declare class Segments {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateSegmentOptions, options?: CreateSegmentRequestOptions): Promise<CreateSegmentResponse>;
    list(options?: ListSegmentsOptions): Promise<ListSegmentsResponse>;
    get(id: string): Promise<GetSegmentResponse>;
    remove(id: string): Promise<RemoveSegmentResponse>;
}

interface Template {
    id: string;
    name: string;
    subject: string | null;
    html: string;
    text: string | null;
    status: 'draft' | 'published';
    variables: TemplateVariable[] | null;
    alias: string | null;
    from: string | null;
    reply_to: string[] | null;
    published_at: string | null;
    created_at: string;
    updated_at: string;
    has_unpublished_versions: boolean;
    current_version_id: string;
}
interface TemplateVariable {
    key: string;
    fallback_value: string | number | null;
    type: 'string' | 'number';
    created_at: string;
    updated_at: string;
}

type TemplateContentCreationOptions = RequireAtLeastOne<{
    html: string;
    react: React.ReactNode;
}>;
type TemplateVariableCreationOptions = Pick<TemplateVariable, 'key' | 'type'> & ({
    type: 'string';
    fallbackValue?: string | null;
} | {
    type: 'number';
    fallbackValue?: number | null;
});
type TemplateOptionalFieldsForCreation = Partial<Pick<Template, 'subject' | 'text' | 'alias' | 'from'>> & {
    replyTo?: string[] | string;
    variables?: TemplateVariableCreationOptions[];
};
type CreateTemplateOptions = Pick<Template, 'name'> & TemplateOptionalFieldsForCreation & TemplateContentCreationOptions;
interface CreateTemplateResponseSuccess extends Pick<Template, 'id'> {
    object: 'template';
}
type CreateTemplateResponse = Response<CreateTemplateResponseSuccess>;

interface DuplicateTemplateResponseSuccess extends Pick<Template, 'id'> {
    object: 'template';
}
type DuplicateTemplateResponse = Response<DuplicateTemplateResponseSuccess>;

interface PublishTemplateResponseSuccess extends Pick<Template, 'id'> {
    object: 'template';
}
type PublishTemplateResponse = Response<PublishTemplateResponseSuccess>;

declare class ChainableTemplateResult<T extends CreateTemplateResponse | DuplicateTemplateResponse> implements PromiseLike<T> {
    private readonly promise;
    private readonly publishFn;
    constructor(promise: Promise<T>, publishFn: (id: string) => Promise<PublishTemplateResponse>);
    then<TResult1 = T, TResult2 = never>(onfulfilled?: ((value: T) => TResult1 | PromiseLike<TResult1>) | null, onrejected?: ((reason: unknown) => TResult2 | PromiseLike<TResult2>) | null): PromiseLike<TResult1 | TResult2>;
    publish(): Promise<PublishTemplateResponse>;
}

interface GetTemplateResponseSuccess extends Template {
    object: 'template';
}
type GetTemplateResponse = Response<GetTemplateResponseSuccess>;

interface TemplateListItem extends Pick<Template, 'id' | 'name' | 'created_at' | 'updated_at' | 'status' | 'published_at' | 'alias'> {
}
interface ListTemplatesResponseSuccess {
    object: 'list';
    data: TemplateListItem[];
    has_more: boolean;
}
type ListTemplatesResponse = Response<ListTemplatesResponseSuccess>;

interface RemoveTemplateResponseSuccess {
    object: 'template';
    id: string;
    deleted: boolean;
}
type RemoveTemplateResponse = Response<RemoveTemplateResponseSuccess>;

type TemplateVariableUpdateOptions = Pick<TemplateVariable, 'key' | 'type'> & ({
    type: 'string';
    fallbackValue?: string | null;
} | {
    type: 'number';
    fallbackValue?: number | null;
});
interface UpdateTemplateOptions extends Partial<Pick<Template, 'name' | 'subject' | 'html' | 'text' | 'from' | 'alias'>> {
    variables?: TemplateVariableUpdateOptions[];
    replyTo?: string[] | string;
}
interface UpdateTemplateResponseSuccess extends Pick<Template, 'id'> {
    object: 'template';
}
type UpdateTemplateResponse = Response<UpdateTemplateResponseSuccess>;

declare class Templates {
    private readonly resend;
    private renderAsync?;
    constructor(resend: Resend);
    create(payload: CreateTemplateOptions): ChainableTemplateResult<CreateTemplateResponse>;
    private performCreate;
    remove(identifier: string): Promise<RemoveTemplateResponse>;
    get(identifier: string): Promise<GetTemplateResponse>;
    list(options?: PaginationOptions): Promise<ListTemplatesResponse>;
    duplicate(identifier: string): ChainableTemplateResult<DuplicateTemplateResponse>;
    publish(identifier: string): Promise<PublishTemplateResponse>;
    update(identifier: string, payload: UpdateTemplateOptions): Promise<UpdateTemplateResponse>;
}

interface Topic {
    id: string;
    name: string;
    description?: string;
    defaultSubscription: 'opt_in' | 'opt_out';
    created_at: string;
}

interface CreateTopicOptions {
    name: string;
    description?: string;
    defaultSubscription: 'opt_in' | 'opt_out';
}
type CreateTopicResponseSuccess = Pick<Topic, 'id'>;
type CreateTopicResponse = Response<CreateTopicResponseSuccess>;

type GetTopicResponseSuccess = Topic;
type GetTopicResponse = Response<GetTopicResponseSuccess>;

interface ListTopicsResponseSuccess {
    data: Topic[];
}
type ListTopicsResponse = Response<ListTopicsResponseSuccess>;

type RemoveTopicResponseSuccess = Pick<Topic, 'id'> & {
    object: 'topic';
    deleted: boolean;
};
type RemoveTopicResponse = Response<RemoveTopicResponseSuccess>;

interface UpdateTopicOptions {
    id: string;
    name?: string;
    description?: string;
}
type UpdateTopicResponseSuccess = Pick<Topic, 'id'>;
type UpdateTopicResponse = Response<UpdateTopicResponseSuccess>;

declare class Topics {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateTopicOptions): Promise<CreateTopicResponse>;
    list(): Promise<ListTopicsResponse>;
    get(id: string): Promise<GetTopicResponse>;
    update(payload: UpdateTopicOptions): Promise<UpdateTopicResponse>;
    remove(id: string): Promise<RemoveTopicResponse>;
}

type WebhookEvent = 'email.sent' | 'email.delivered' | 'email.delivery_delayed' | 'email.complained' | 'email.bounced' | 'email.opened' | 'email.clicked' | 'email.received' | 'email.failed' | 'contact.created' | 'contact.updated' | 'contact.deleted' | 'domain.created' | 'domain.updated' | 'domain.deleted';

interface CreateWebhookOptions {
    endpoint: string;
    events: WebhookEvent[];
}
interface CreateWebhookRequestOptions extends PostOptions {
}
interface CreateWebhookResponseSuccess {
    object: 'webhook';
    id: string;
    signing_secret: string;
}
type CreateWebhookResponse = Response<CreateWebhookResponseSuccess>;

interface GetWebhookResponseSuccess {
    object: 'webhook';
    id: string;
    created_at: string;
    status: 'enabled' | 'disabled';
    endpoint: string;
    events: WebhookEvent[] | null;
    signing_secret: string;
}
type GetWebhookResponse = Response<GetWebhookResponseSuccess>;

type ListWebhooksOptions = PaginationOptions;
interface Webhook {
    id: string;
    endpoint: string;
    created_at: string;
    status: 'enabled' | 'disabled';
    events: WebhookEvent[] | null;
}
type ListWebhooksResponseSuccess = {
    object: 'list';
    has_more: boolean;
    data: Webhook[];
};
type ListWebhooksResponse = Response<ListWebhooksResponseSuccess>;

type RemoveWebhookResponseSuccess = Pick<Webhook, 'id'> & {
    object: 'webhook';
    deleted: boolean;
};
type RemoveWebhookResponse = Response<RemoveWebhookResponseSuccess>;

interface UpdateWebhookOptions {
    endpoint?: string;
    events?: WebhookEvent[];
    status?: 'enabled' | 'disabled';
}
interface UpdateWebhookResponseSuccess {
    object: 'webhook';
    id: string;
}
type UpdateWebhookResponse = Response<UpdateWebhookResponseSuccess>;

interface Headers {
    id: string;
    timestamp: string;
    signature: string;
}
interface VerifyWebhookOptions {
    payload: string;
    headers: Headers;
    webhookSecret: string;
}
declare class Webhooks {
    private readonly resend;
    constructor(resend: Resend);
    create(payload: CreateWebhookOptions, options?: CreateWebhookRequestOptions): Promise<CreateWebhookResponse>;
    get(id: string): Promise<GetWebhookResponse>;
    list(options?: ListWebhooksOptions): Promise<ListWebhooksResponse>;
    update(id: string, payload: UpdateWebhookOptions): Promise<UpdateWebhookResponse>;
    remove(id: string): Promise<RemoveWebhookResponse>;
    verify(payload: VerifyWebhookOptions): unknown;
}

declare class Resend {
    readonly key?: string | undefined;
    private readonly headers;
    readonly apiKeys: ApiKeys;
    readonly segments: Segments;
    /**
     * @deprecated Use segments instead
     */
    readonly audiences: Segments;
    readonly batch: Batch;
    readonly broadcasts: Broadcasts;
    readonly contacts: Contacts;
    readonly contactProperties: ContactProperties;
    readonly domains: Domains;
    readonly emails: Emails;
    readonly webhooks: Webhooks;
    readonly templates: Templates;
    readonly topics: Topics;
    constructor(key?: string | undefined);
    fetchRequest<T>(path: string, options?: {}): Promise<Response<T>>;
    post<T>(path: string, entity?: unknown, options?: PostOptions & IdempotentRequest): Promise<Response<T>>;
    get<T>(path: string, options?: GetOptions): Promise<Response<T>>;
    put<T>(path: string, entity: unknown, options?: PutOptions): Promise<Response<T>>;
    patch<T>(path: string, entity: unknown, options?: PatchOptions): Promise<Response<T>>;
    delete<T>(path: string, query?: unknown): Promise<Response<T>>;
}

export { type ApiKey, type Attachment$1 as Attachment, type Broadcast, type CancelEmailResponse, type CancelEmailResponseSuccess, type Contact, type CreateApiKeyOptions, type CreateApiKeyRequestOptions, type CreateApiKeyResponse, type CreateApiKeyResponseSuccess, type CreateBatchOptions, type CreateBatchRequestOptions, type CreateBatchResponse, type CreateBatchSuccessResponse, type CreateBroadcastOptions, type CreateBroadcastRequestOptions, type CreateBroadcastResponse, type CreateBroadcastResponseSuccess, type CreateContactOptions, type CreateContactRequestOptions, type CreateContactResponse, type CreateContactResponseSuccess, type CreateDomainOptions, type CreateDomainRequestOptions, type CreateDomainResponse, type CreateDomainResponseSuccess, type CreateEmailOptions, type CreateEmailRequestOptions, type CreateEmailResponse, type CreateEmailResponseSuccess, type CreateSegmentOptions, type CreateSegmentRequestOptions, type CreateSegmentResponse, type CreateSegmentResponseSuccess, type CreateWebhookOptions, type CreateWebhookRequestOptions, type CreateWebhookResponse, type CreateWebhookResponseSuccess, type Domain, type DomainApiOptions, type DomainDkimRecord, type DomainNameservers, type DomainRecords, type DomainRegion, type DomainSpfRecord, type DomainStatus, type EmailApiAttachment, type EmailApiOptions, type ErrorResponse, type GetAttachmentOptions, type GetAttachmentResponse, type GetAttachmentResponseSuccess, type GetBroadcastResponse, type GetBroadcastResponseSuccess, type GetContactOptions, type GetContactResponse, type GetContactResponseSuccess, type GetDomainResponse, type GetDomainResponseSuccess, type GetEmailResponse, type GetEmailResponseSuccess, type GetOptions, type GetReceivingEmailResponse, type GetReceivingEmailResponseSuccess, type GetSegmentResponse, type GetSegmentResponseSuccess, type GetWebhookResponse, type GetWebhookResponseSuccess, type IdempotentRequest, type ListApiKeysOptions, type ListApiKeysResponse, type ListApiKeysResponseSuccess, type ListAttachmentsApiResponse, type ListAttachmentsOptions, type ListAttachmentsResponse, type ListAttachmentsResponseSuccess, type ListAudienceContactsOptions, type ListBroadcastsOptions, type ListBroadcastsResponse, type ListBroadcastsResponseSuccess, type ListContactsOptions, type ListContactsResponse, type ListContactsResponseSuccess, type ListDomainsOptions, type ListDomainsResponse, type ListDomainsResponseSuccess, type ListReceivingEmail, type ListReceivingEmailsOptions, type ListReceivingEmailsResponse, type ListReceivingEmailsResponseSuccess, type ListSegmentsOptions, type ListSegmentsResponse, type ListSegmentsResponseSuccess, type ListWebhooksOptions, type ListWebhooksResponse, type ListWebhooksResponseSuccess, type PaginatedData, type PaginationOptions, type PatchOptions, type PostOptions, type PutOptions, type ReceivingRecord, type RemoveApiKeyResponse, type RemoveApiKeyResponseSuccess, type RemoveBroadcastResponse, type RemoveBroadcastResponseSuccess, type RemoveContactOptions, type RemoveContactsResponse, type RemoveContactsResponseSuccess, type RemoveDomainsResponse, type RemoveDomainsResponseSuccess, type RemoveSegmentResponse, type RemoveSegmentResponseSuccess, type RemoveWebhookResponse, type RemoveWebhookResponseSuccess, type RequireAtLeastOne, Resend, type Segment, type SelectingField, type SendBroadcastOptions, type SendBroadcastRequestOptions, type SendBroadcastResponse, type SendBroadcastResponseSuccess, type Tag, type UpdateBroadcastOptions, type UpdateBroadcastResponse, type UpdateBroadcastResponseSuccess, type UpdateContactOptions, type UpdateContactResponse, type UpdateContactResponseSuccess, type UpdateDomainsOptions, type UpdateDomainsResponse, type UpdateDomainsResponseSuccess, type UpdateEmailOptions, type UpdateEmailResponse, type UpdateEmailResponseSuccess, type UpdateWebhookOptions, type UpdateWebhookResponse, type UpdateWebhookResponseSuccess, type VerifyDomainsResponse, type VerifyDomainsResponseSuccess, type Webhook, type WebhookEvent };
